// Copyright 2014 Google Inc. All Rights Reserved.

#include "common.h"
#include "SensorSource.h"

#define PROTO_FIRST_MEMBER(proto, member) \
    (likely(proto.member##_size()) ? proto.mutable_##member(0) : proto.add_##member())

#define PROTO_FIRST_MEMBER_CLEAR(dest, proto, member) \
do {                                                  \
    if (likely(proto.member##_size())) {              \
        dest = proto.mutable_##member(0);             \
        dest->Clear();                                \
    } else {                                          \
        dest = proto.add_##member();                  \
    }                                                 \
} while (0)

#define PROTO_SET_OPTIONAL(proto, member, cond, val)    \
do {                                                    \
    if (cond) {                                         \
        (proto)->set_##member(val);                     \
    }                                                   \
} while (0)

#define CONDITIONAL_CACHE_FETCH(update_period, dest, src, type, member, modified)     \
do {                                                                                  \
    if (update_period != SENSOR_UPDATE_PERIOD_NEVER) {                                \
        Autolock m(&mCacheLock);                                                      \
        if (mLastValuesValid[type]) {                                                 \
            dest.add_##member()->Swap(src.mutable_##member(0));                       \
            modified = true;                                                          \
            mLastValuesValid[type] = false;                                           \
        }                                                                             \
    }                                                                                 \
} while (0)

#define PROTO_SET_REPEATED(proto, member, ptr, size)            \
do {                                                            \
    if (ptr != NULL && size > 0) {                              \
        proto->clear_##member();                                \
        for (int i = 0; i < size; i++) {                        \
            proto->add_##member(ptr[i]);                        \
        }                                                       \
    }                                                           \
} while (0)

void SensorSource::addDiscoveryInfo(ServiceDiscoveryResponse *sdr) {
    Service* service = sdr->add_services();
    service->set_id(id());
    SensorSourceService* sss = service->mutable_sensor_source_service();

    for(set<int32_t>::iterator it = mSensors.begin(); it != mSensors.end(); ++it) {
        SensorSourceService_Sensor* sensor = sss->add_sensors();
        sensor->set_sensor_type((SensorType) *it);
    }
    sss->set_location_characterization(mLocationCharacterization);
}

int SensorSource::handleSensorRequest(const SensorRequest& req) {
    int ret = STATUS_SUCCESS;

    if (mSensors.find(req.type()) == mSensors.end()) {
        ret = STATUS_INVALID_SENSOR;
        sendSensorResponse(ret);
        return ret;
    }

    SensorBatch batch;
    bool shouldSend = false;
    switch (req.type()) {
        case SENSOR_LOCATION:
            mLocationUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mLocationUpdatePeriod, batch, mLastValues,
                    SENSOR_LOCATION, location_data, shouldSend);
            break;
        case SENSOR_COMPASS:
            mCompassUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mCompassUpdatePeriod, batch, mLastValues,
                    SENSOR_COMPASS, compass_data, shouldSend);
            break;
        case SENSOR_SPEED:
            mSpeedUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mSpeedUpdatePeriod, batch, mLastValues,
                    SENSOR_SPEED, speed_data, shouldSend);
            break;
        case SENSOR_RPM:
            mRpmUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mRpmUpdatePeriod, batch, mLastValues,
                    SENSOR_RPM, rpm_data, shouldSend);
            break;
        case SENSOR_ODOMETER:
            mOdometerUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mOdometerUpdatePeriod, batch, mLastValues,
                    SENSOR_ODOMETER, odometer_data, shouldSend);
            break;
        case SENSOR_FUEL:
            mFuelUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mFuelUpdatePeriod, batch, mLastValues,
                    SENSOR_FUEL, fuel_data, shouldSend);
            break;
        case SENSOR_PARKING_BRAKE:
            mParkingBrakeUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mParkingBrakeUpdatePeriod, batch, mLastValues,
                    SENSOR_PARKING_BRAKE, parking_brake_data, shouldSend);
            break;
        case SENSOR_GEAR:
            mGearUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mGearUpdatePeriod, batch, mLastValues,
                    SENSOR_GEAR, gear_data, shouldSend);
            break;
        case SENSOR_OBDII_DIAGNOSTIC_CODE:
            mDiagnosticCodeUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mDiagnosticCodeUpdatePeriod, batch, mLastValues,
                    SENSOR_OBDII_DIAGNOSTIC_CODE, diagnostics_data, shouldSend);
            break;
        case SENSOR_NIGHT_MODE:
            mNightModeUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mNightModeUpdatePeriod, batch, mLastValues,
                    SENSOR_NIGHT_MODE, night_mode_data, shouldSend);
            break;
        case SENSOR_ENVIRONMENT_DATA:
            mEnvironmentDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mEnvironmentDataUpdatePeriod, batch, mLastValues,
                    SENSOR_ENVIRONMENT_DATA, environment_data, shouldSend);
            break;
        case SENSOR_HVAC_DATA:
            mHvacDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mHvacDataUpdatePeriod, batch, mLastValues,
                    SENSOR_HVAC_DATA, hvac_data, shouldSend);
            break;
        case SENSOR_DRIVING_STATUS_DATA:
            mDrivingStatusDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mDrivingStatusDataUpdatePeriod, batch, mLastValues,
                    SENSOR_DRIVING_STATUS_DATA, driving_status_data, shouldSend);
            break;
        case SENSOR_DEAD_RECKONING_DATA:
            mDeadReckoningDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mDeadReckoningDataUpdatePeriod, batch, mLastValues,
                    SENSOR_DEAD_RECKONING_DATA, dead_reckoning_data, shouldSend);
            break;
        case SENSOR_PASSENGER_DATA:
            mPassengerDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mPassengerDataUpdatePeriod, batch, mLastValues,
                    SENSOR_PASSENGER_DATA, passenger_data, shouldSend);
            break;
        case SENSOR_DOOR_DATA:
            mDoorDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mDoorDataUpdatePeriod, batch, mLastValues,
                    SENSOR_DOOR_DATA, door_data, shouldSend);
            break;
        case SENSOR_LIGHT_DATA:
            mLightDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mLightDataUpdatePeriod, batch, mLastValues,
                    SENSOR_LIGHT_DATA, light_data, shouldSend);
            break;
        case SENSOR_TIRE_PRESSURE_DATA:
            mTirePressureDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mTirePressureDataUpdatePeriod, batch, mLastValues,
                    SENSOR_TIRE_PRESSURE_DATA, tire_pressure_data, shouldSend);
            break;
        case SENSOR_ACCELEROMETER_DATA:
            mAccelerometerDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mAccelerometerDataUpdatePeriod, batch, mLastValues,
                    SENSOR_ACCELEROMETER_DATA, accelerometer_data, shouldSend);
            break;
        case SENSOR_GYROSCOPE_DATA:
            mGyroscopeDataUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mGyroscopeDataUpdatePeriod, batch, mLastValues,
                    SENSOR_GYROSCOPE_DATA, gyroscope_data, shouldSend);
            break;
        case SENSOR_GPS_SATELLITE_DATA:
            mGpsSatelliteUpdatePeriod = req.min_update_period();
            CONDITIONAL_CACHE_FETCH(mGpsSatelliteUpdatePeriod, batch, mLastValues,
                    SENSOR_GPS_SATELLITE_DATA, gps_satellite_data, shouldSend);
            break;
        default:
            ret = STATUS_INVALID_SENSOR;
    }
    sendSensorResponse(ret);
    if (shouldSend) {
        sendSensorBatch(batch);
    }
    return ret;
}

void SensorSource::sendSensorResponse(int32_t status) {
    SensorResponse resp;
    resp.set_status(static_cast<MessageStatus>(status));
    IoBuffer buf;
    mRouter->marshallProto(SENSOR_MESSAGE_RESPONSE, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

int SensorSource::routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    SensorRequest req;
    int ret = STATUS_UNEXPECTED_MESSAGE;
    switch (type) {
    case SENSOR_MESSAGE_REQUEST:
        uint8_t* ptr = (uint8_t*) msg->raw() + sizeof(uint16_t);
        size_t len = msg->size() - sizeof(uint16_t);
        if (PARSE_PROTO(req, ptr, len)) {
            ret = handleSensorRequest(req);
        }
    }
    return ret;
}

void SensorSource::tick(uint64_t timestamp) {
    // TODO - Implement batching.
}

void SensorSource::sendSensorBatch(const SensorBatch& batch) {
    IoBuffer buf;
    mRouter->marshallProto(SENSOR_MESSAGE_BATCH, batch, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

static void populateLocationData(LocationData* data,
        uint64_t timestamp, int32_t latitudeE7, int32_t longitudeE7,
        bool hasAccuracy, int32_t accuracyE3, bool hasAltitude, int32_t altitudeE2,
        bool hasSpeed, int32_t speedE3, bool hasBearing, int32_t bearingE6) {
    data->set_timestamp(timestamp);
    data->set_latitude_e7(latitudeE7);
    data->set_longitude_e7(longitudeE7);
    PROTO_SET_OPTIONAL(data, accuracy_e3, hasAccuracy, accuracyE3);
    PROTO_SET_OPTIONAL(data, altitude_e2, hasAltitude, altitudeE2);
    PROTO_SET_OPTIONAL(data, speed_e3, hasSpeed, speedE3);
    PROTO_SET_OPTIONAL(data, bearing_e6, hasBearing, bearingE6);
}

int SensorSource::reportLocationData(uint64_t timestamp, int32_t latitudeE7, int32_t longitudeE7,
        bool hasAccuracy, int32_t accuracyE3, bool hasAltitude, int32_t altitudeE2,
        bool hasSpeed, int32_t speedE3, bool hasBearing, int32_t bearingE6) {
    LocationData* data;
    if (mLocationUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_location_data();
        populateLocationData(data, timestamp, latitudeE7, longitudeE7, hasAccuracy, accuracyE3,
                hasAltitude, altitudeE2, hasSpeed, speedE3, hasBearing, bearingE6);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, location_data);
        populateLocationData(data, timestamp, latitudeE7, longitudeE7, hasAccuracy, accuracyE3,
                hasAltitude, altitudeE2, hasSpeed, speedE3, hasBearing, bearingE6);
        mLastValuesValid[SENSOR_LOCATION] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportCompassData(int32_t bearingE6) {
    return reportCompassData3D(bearingE6, false, 0, false, 0);
}

int SensorSource::reportCompassData3D(int32_t bearingE6, bool hasPitch, int32_t pitchE6,
                                      bool hasRoll, int32_t rollE6) {
    CompassData* data;
    if (mCompassUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_compass_data();
        data->set_bearing_e6(bearingE6);
        PROTO_SET_OPTIONAL(data, pitch_e6, hasPitch, pitchE6);
        PROTO_SET_OPTIONAL(data, roll_e6, hasRoll, rollE6);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, compass_data);
        data->set_bearing_e6(bearingE6);
        PROTO_SET_OPTIONAL(data, pitch_e6, hasPitch, pitchE6);
        PROTO_SET_OPTIONAL(data, roll_e6, hasRoll, rollE6);
        mLastValuesValid[SENSOR_COMPASS] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportSpeedData(int32_t speedE3, bool hasCruiseEngaged, bool cruiseEngaged,
        bool hasCruiseSetSpeed, int32_t cruiseSetSpeed) {
    if (mSpeedUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        SpeedData* data = batch.add_speed_data();
        data->set_speed_e3(speedE3);
        PROTO_SET_OPTIONAL(data, cruise_engaged, hasCruiseEngaged, cruiseEngaged);
        PROTO_SET_OPTIONAL(data, cruise_set_speed, hasCruiseSetSpeed, cruiseSetSpeed);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, speed_data)->set_speed_e3(speedE3);
        mLastValuesValid[SENSOR_SPEED] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportRpmData(int32_t rpmE3) {
    if (mRpmUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        RpmData* data = batch.add_rpm_data();
        data->set_rpm_e3(rpmE3);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, rpm_data)->set_rpm_e3(rpmE3);
        mLastValuesValid[SENSOR_RPM] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportOdometerData(int32_t kmsE1, bool hasTripKms, int32_t tripKmsE1) {
    if (mOdometerUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        OdometerData* data = batch.add_odometer_data();
        PROTO_SET_OPTIONAL(data, trip_kms_e1, hasTripKms, tripKmsE1);
        data->set_kms_e1(kmsE1);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, odometer_data)->set_kms_e1(kmsE1);
        mLastValuesValid[SENSOR_ODOMETER] = true;
    }
    return STATUS_SUCCESS;
}


void SensorSource::populateFuelData(FuelData* data, bool hasFuelRemainingPercent,
        int32_t fuelRemainingPercent, bool hasRangeKm, int32_t rangeKm,
        bool hasLowFuelWarning, bool lowFuelWarning) {
    PROTO_SET_OPTIONAL(data, fuel_level, hasFuelRemainingPercent, fuelRemainingPercent);
    PROTO_SET_OPTIONAL(data, range, hasRangeKm, rangeKm);
    PROTO_SET_OPTIONAL(data, low_fuel_warning, hasLowFuelWarning, lowFuelWarning);
}

int SensorSource::reportFuelData(bool hasFuelRemainingPercent, int32_t fuelRemainingPercent,
        bool hasRangeKm, int32_t rangeKm, bool hasLowFuelWarning, bool lowFuelWarning) {
    FuelData* data;
    if (mFuelUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_fuel_data();
        populateFuelData(data, hasFuelRemainingPercent, fuelRemainingPercent, hasRangeKm, rangeKm,
                hasLowFuelWarning, lowFuelWarning);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, fuel_data);
        populateFuelData(data, hasFuelRemainingPercent, fuelRemainingPercent, hasRangeKm, rangeKm,
                hasLowFuelWarning, lowFuelWarning);
        mLastValuesValid[SENSOR_FUEL] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportParkingBrakeData(bool engaged) {
    if (mParkingBrakeUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        ParkingBrakeData* data = batch.add_parking_brake_data();
        data->set_parking_brake(engaged);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, parking_brake_data)->set_parking_brake(engaged);
        mLastValuesValid[SENSOR_PARKING_BRAKE] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportGearData(int gear) {
    if (mGearUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        GearData* data = batch.add_gear_data();
        data->set_gear((Gear) gear);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, gear_data)->set_gear((Gear) gear);
        mLastValuesValid[SENSOR_GEAR] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportDiagnosticsData(uint8_t *data, size_t len) {
    DiagnosticsData* diagData;
    if (mDiagnosticCodeUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        diagData = batch.add_diagnostics_data();
        diagData->set_dtc(data, len);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(diagData, mLastValues, diagnostics_data);
        diagData->set_dtc(data, len);
        mLastValuesValid[SENSOR_OBDII_DIAGNOSTIC_CODE] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportNightModeData(bool night_mode) {
    if (mNightModeUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        NightModeData* data = batch.add_night_mode_data();
        data->set_night_mode(night_mode);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, night_mode_data)->set_night_mode(night_mode);
        mLastValuesValid[SENSOR_NIGHT_MODE] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportEnvironmentData(bool hasTemperature, int32_t temperatureE3,
        bool hasPressure, int32_t pressureE3, bool hasRain, int32_t rain) {
    EnvironmentData* data;
    if (mEnvironmentDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_environment_data();
        PROTO_SET_OPTIONAL(data, temperature_e3, hasTemperature, temperatureE3);
        PROTO_SET_OPTIONAL(data, pressure_e3, hasPressure, pressureE3);
        PROTO_SET_OPTIONAL(data, rain, hasRain, rain);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, environment_data);
        PROTO_SET_OPTIONAL(data, temperature_e3, hasTemperature, temperatureE3);
        PROTO_SET_OPTIONAL(data, pressure_e3, hasPressure, pressureE3);
        mLastValuesValid[SENSOR_ENVIRONMENT_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportHvacData(bool hasTargetTemperature, int32_t targetTemperatureE3,
        bool hasCurrentTemperature, int32_t currentTemperatureE3) {
    HvacData* data;
    if (mHvacDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        sendSensorBatch(batch);
        data = batch.add_hvac_data();
        PROTO_SET_OPTIONAL(data, target_temperature_e3, hasTargetTemperature, targetTemperatureE3);
        PROTO_SET_OPTIONAL(data, current_temperature_e3, \
                hasCurrentTemperature, currentTemperatureE3);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, hvac_data);
        PROTO_SET_OPTIONAL(data, target_temperature_e3, hasTargetTemperature, targetTemperatureE3);
        PROTO_SET_OPTIONAL(data, current_temperature_e3,
                hasCurrentTemperature, currentTemperatureE3);
        mLastValuesValid[SENSOR_HVAC_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportDrivingStatusData(int32_t drivingStatus) {
    if (mDrivingStatusDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        DrivingStatusData* data = batch.add_driving_status_data();
        data->set_status(drivingStatus);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, driving_status_data)->set_status(
                (DrivingStatus) drivingStatus);
        mLastValuesValid[SENSOR_DRIVING_STATUS_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportDeadReckoningData(bool hasSteeringAngle, int32_t steeringAngleE1,
        const int32_t* wheelSpeedE3, int32_t wheelSpeedEntries) {
    DeadReckoningData* data;
    if (mDeadReckoningDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_dead_reckoning_data();
        PROTO_SET_OPTIONAL(data, steering_angle_e1, hasSteeringAngle, steeringAngleE1);
        PROTO_SET_REPEATED(data, wheel_speed_e3, wheelSpeedE3, wheelSpeedEntries);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, dead_reckoning_data);
        PROTO_SET_OPTIONAL(data, steering_angle_e1, hasSteeringAngle, steeringAngleE1);
        PROTO_SET_REPEATED(data, wheel_speed_e3, wheelSpeedE3, wheelSpeedEntries);
        mLastValuesValid[SENSOR_DEAD_RECKONING_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportPassengerData(bool passengerPresent) {
    if (mPassengerDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        PassengerData* data = batch.add_passenger_data();
        data->set_passenger_present(passengerPresent);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER(mLastValues, passenger_data)->set_passenger_present(passengerPresent);
        mLastValuesValid[SENSOR_PASSENGER_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportDoorData(bool hasHoodOpen, bool hoodOpen,
        bool hasTrunkOpen, bool trunkOpen,
        const bool* doorOpen, int32_t doorOpenEntries) {
    DoorData* data;
    if (mDoorDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_door_data();
        PROTO_SET_OPTIONAL(data, hood_open, hasHoodOpen, hoodOpen);
        PROTO_SET_OPTIONAL(data, trunk_open, hasTrunkOpen, trunkOpen);
        PROTO_SET_REPEATED(data, door_open, doorOpen, doorOpenEntries);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, door_data);
        PROTO_SET_OPTIONAL(data, hood_open, hasHoodOpen, hoodOpen);
        PROTO_SET_OPTIONAL(data, trunk_open, hasTrunkOpen, trunkOpen);
        PROTO_SET_REPEATED(data, door_open, doorOpen, doorOpenEntries);
        mLastValuesValid[SENSOR_DOOR_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportLightData(bool hasHeadLightState, int headLightState,
        bool hasTurnIndicatorState, int turnIndicatorState,
        bool hasHazardLightsOn, bool hazardLightsOn) {
    LightData* data;
    if (mLightDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_light_data();
        PROTO_SET_OPTIONAL(data, head_light_state, hasHeadLightState,
                (HeadLightState) headLightState);
        PROTO_SET_OPTIONAL(data, turn_indicator_state, hasTurnIndicatorState,
                (TurnIndicatorState) turnIndicatorState);
        PROTO_SET_OPTIONAL(data, hazard_lights_on, hasHazardLightsOn, hazardLightsOn);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, light_data);
        PROTO_SET_OPTIONAL(data, head_light_state, hasHeadLightState,
                (HeadLightState) headLightState);
        PROTO_SET_OPTIONAL(data, turn_indicator_state, hasTurnIndicatorState,
                (TurnIndicatorState) turnIndicatorState);
        PROTO_SET_OPTIONAL(data, hazard_lights_on, hasHazardLightsOn, hazardLightsOn);
        mLastValuesValid[SENSOR_LIGHT_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportTirePressureData(const int32_t* tirePressuresE2, int32_t numEntries) {
    if (mTirePressureDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        TirePressureData* data = batch.add_tire_pressure_data();
        PROTO_SET_REPEATED(data, tire_pressures_e2, tirePressuresE2, numEntries);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        TirePressureData* data = PROTO_FIRST_MEMBER(mLastValues, tire_pressure_data);
        PROTO_SET_REPEATED(data, tire_pressures_e2, tirePressuresE2, numEntries);
        mLastValuesValid[SENSOR_TIRE_PRESSURE_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportAccelerometerData(bool hasAccelerationX, int32_t xAccelerationE3,
        bool hasAccelerationY, int32_t yAccelerationE3, bool hasAccelerationZ,
        int32_t zAccelerationE3) {
    AccelerometerData* data;
    if (mAccelerometerDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_accelerometer_data();
        PROTO_SET_OPTIONAL(data, acceleration_x_e3, hasAccelerationX, xAccelerationE3);
        PROTO_SET_OPTIONAL(data, acceleration_y_e3, hasAccelerationY, yAccelerationE3);
        PROTO_SET_OPTIONAL(data, acceleration_z_e3, hasAccelerationZ, zAccelerationE3);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, accelerometer_data);
        PROTO_SET_OPTIONAL(data, acceleration_x_e3, hasAccelerationX, xAccelerationE3);
        PROTO_SET_OPTIONAL(data, acceleration_y_e3, hasAccelerationY, yAccelerationE3);
        PROTO_SET_OPTIONAL(data, acceleration_z_e3, hasAccelerationZ, zAccelerationE3);
        mLastValuesValid[SENSOR_ACCELEROMETER_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportGyroscopeData(bool hasRotationSpeedX, int32_t xRotationSpeedE3,
        bool hasRotationSpeedY, int32_t yRotationSpeedE3, bool hasRotationSpeedZ,
        int32_t zRotationSpeedE3) {
    GyroscopeData* data;
    if (mGyroscopeDataUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_gyroscope_data();
        PROTO_SET_OPTIONAL(data, rotation_speed_x_e3, hasRotationSpeedX, xRotationSpeedE3);
        PROTO_SET_OPTIONAL(data, rotation_speed_y_e3, hasRotationSpeedY, yRotationSpeedE3);
        PROTO_SET_OPTIONAL(data, rotation_speed_z_e3, hasRotationSpeedZ, zRotationSpeedE3);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, gyroscope_data);
        PROTO_SET_OPTIONAL(data, rotation_speed_x_e3, hasRotationSpeedX, xRotationSpeedE3);
        PROTO_SET_OPTIONAL(data, rotation_speed_y_e3, hasRotationSpeedY, yRotationSpeedE3);
        PROTO_SET_OPTIONAL(data, rotation_speed_z_e3, hasRotationSpeedZ, zRotationSpeedE3);
        mLastValuesValid[SENSOR_GYROSCOPE_DATA] = true;
    }
    return STATUS_SUCCESS;
}

static void fillGpsSatellites(GpsSatelliteData* data, int32_t numberInUse, bool hasNumberInView,
        int32_t numberInView, const int32_t *prns, const int32_t *snrsE3, const bool *usedInFix,
        const int32_t *azimuthsE3, const int32_t *elevationsE3) {
    data->set_number_in_use(numberInUse);
    PROTO_SET_OPTIONAL(data, number_in_view, hasNumberInView, numberInView);
    if (hasNumberInView && prns != NULL && snrsE3 != NULL && usedInFix != NULL) {
        for (int i = 0; i < numberInView; ++i) {
            GpsSatellite* satellite = data->add_satellites();
            satellite->set_prn(prns[i]);
            satellite->set_snr_e3(snrsE3[i]);
            satellite->set_used_in_fix(usedInFix[i]);
            if (azimuthsE3 != NULL && elevationsE3 != NULL) {
                satellite->set_azimuth_e3(azimuthsE3[i]);
                satellite->set_elevation_e3(elevationsE3[i]);
            }
        }
    }
}

int SensorSource::reportGpsSatelliteData(int32_t numberInUse, bool hasNumberInView,
        int32_t numberInView, const int32_t *prns, const int32_t *snrsE3, const bool *usedInFix,
        const int32_t *azimuthsE3, const int32_t *elevationsE3) {
    GpsSatelliteData* data;
    if (mGpsSatelliteUpdatePeriod != SENSOR_UPDATE_PERIOD_NEVER) {
        SensorBatch batch;
        data = batch.add_gps_satellite_data();
        fillGpsSatellites(data, numberInUse, hasNumberInView, numberInView, prns, snrsE3,
                usedInFix, azimuthsE3, elevationsE3);
        sendSensorBatch(batch);
    } else {
        Autolock l(&mCacheLock);
        PROTO_FIRST_MEMBER_CLEAR(data, mLastValues, gps_satellite_data);
        fillGpsSatellites(data, numberInUse, hasNumberInView, numberInView, prns, snrsE3,
                usedInFix, azimuthsE3, elevationsE3);
        mLastValuesValid[SENSOR_GPS_SATELLITE_DATA] = true;
    }
    return STATUS_SUCCESS;
}

int SensorSource::reportSensorError(int type, int errorType) {
    SensorError errMsg;
    errMsg.set_sensor_type(static_cast<SensorType>(type));
    errMsg.set_sensor_error_type(static_cast<SensorErrorType>(errorType));
    IoBuffer buf;
    mRouter->marshallProto(SENSOR_MESSAGE_ERROR, errMsg, &buf);
    queueOutgoing(buf.raw(), buf.size());
    return STATUS_SUCCESS;
}
